/* ***** BEGIN LICENSE BLOCK *****
 *
 * SmoothWheel: Smart algorithms for smooth scrolling with the mouse wheel.
 * Copyright (C) 2003 Avi Halachmi
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * contact:  avihpit (*at*) yahoo (*dot*) com
 *
 * The Original Code is SmoothWheel.
 *
 * The Initial Developer of the Original Code is Avi Halachmi.
 * Portions created by the Initial Developer are Copyright (C) 2003 Avi Halachmi.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * ***** END LICENSE BLOCK ***** */


/*********************************************************
 *
 * SmoothWheel by Avi Halachmi (avihpit (*at*) yahoo (*dot*) com)
 * SmoothWheel extension for Firefox/Thunderbird/SeaMonkey/SongBird/Flock/Komposer/NV|U
 * SmoothWheel home page: http://smoothwheel.mozdev.org
 * Download latest version: http://downloads.mozdev.org/smoothwheel/smoothwheel.xpi
 * Licence: GPL 2.0
 * Copyright 2003 Avi Halachmi (avihpit (*at*) yahoo (*dot*) com)
 *
 * About:
 * The smoothwheel extension will scroll the page smoothly with the mouse wheel.
 * SmoothWheel utilizes several smart algorithms for smoothness and adaptive behaviour.
 * It replaces the internal Gecko smoothscroll which was introduced on 2003-03-24.
 * Tested platforms: win32, Linux, FreeBsd/i386, Mac OS/X (theoretically should work on all platforms).
 * Not officially supported anymore (deprecated) but may work: Mozilla/Firebird/Phoenix/Netscape/.
 *
 * Features:
 * - Can be disabled easily
 * - Automatically utilizes strong CPUs for smoother scroll.
 * - Very configurable
 * - Scroll steps relative to scrollable area height (configurable).
 * - Advanced Optimized algorithm for a VERY smooth scroll in many scenarios.
 * - Soft-Edge : Lands gracefully on top/bottom of document.
 * - Adaptive-Step : Shorter duration between wheel events -> more pixels/event.
 * - Adaptive-Duration : Shorter duration between wheel events -> scrolls faster.
 * - Modifier keys for small/full page steps (SHIFT and ALT respectively by default).
 * - Timer based operation: Always 'finishes on time'.
 * - Lets the default mouse wheel handler work if smoothwheel can't handle it.
 *
 * Initial release (Avi Halachmi): 22-Mar-2003 (v0.1)
 *
 * modified (Avi Halachmi): 15-Apr-2003 (v0.2):
 *  - Improved Algorithm, Improved Timing, much improved target detection,
 *    Modifier keys, Better configuration and presets, Optimizations.
 *
 * modified (Avi Halachmi): 22-May-2003 (v0.3):
 *  - New target detection code (courtesy of All-In-One-Gestures author), "disable" Modifier key,
 *    soft-edge, adaptive-step, adaptive-duration, works smoothly on view-src and composer as well,
 *    namespace improvements, more event handler optimizations for the default settings, some more presets.
 *
 * modified (Avi Halachmi): 2003-06-05 (v0.31):
 *  - Compatibility with Scroll-Wheel-Navigation at All-In-One-Gestures extension fixed,
 *  - Slight code cleanups.
 *
 * modified (Avi Halachmi): 2003-08-02 (v0.32):
 *  - Works in Mozilla-Thunderbird and Help window (Mozilla/Netscape).
 *  - Improved compatibility with Mozilla-Firebird 0.6.1 and Mozilla-Thunderbird 0.1.
 *
 * modified (Avi Halachmi): 2004-01-27 (v0.33):
 *  - Improved compatibility with latest Moz abd Birds releases (detection code from AIO v0.96)
 *
 * modified (Avi Halachmi): 2004-02-01 (v0.33.1):
 *  - Removed 'handled(normal)' alert if default settings are changed.
 *
 * modified (Avi Halachmi): 2004-02-04 (v0.4):
 *  - Refined algorithms implementation (a bit more responsive)
 *  - Averaged mouse wheel events (better predictability and stability)
 *  - Fixed: occasional misbehaviour if not using a preset.
 *
 * modified (Avi Halachmi): 2004-10-22 (v0.41, 0.42):
 *  - Added compatibility with Firefox 1.0 and Thunderbird 1.0 new Extension-manager API
 *
 * modified (Avi Halachmi): 2004-11-07 (v0.43):
 *  - Added configuration panel
 *
 * modified (Avi Halachmi): 2004-11-07 (v0.43.1.20041107):
 *  - max-version reduced such that it's accepted by update.mozilla.org
 *  - added build date to the version number
 *  - added tooltip for the modifiers section in the configuration panel
 *
 * modified (Avi Halachmi): 2005-05-23 (v0.44.5.20050523):
 *  - added more variables to the configuration panel
 *  - modified prefutils a bit to allow foating point values (as strings)
 *  - more options exposed, better usability by rearrangement of settings and an about panel.
 *  - clean all smoothwheel preferences if upgrading a version
 *
 * modified (Avi Halachmi): 2005-05-27 (v0.44.6.20050527)
 *  - bugfix: on linux only with firefox 1.04 scrolling over a (scrollable) drop-down box scrolled the page instead of the box. fixed.
 *  - bugfix: on linux and windows scrolling over a non-scrollable drop-down box scrolled the page (instead of doing nothing). fixed.
 *
 * modified (Avi Halachmi): 2005-05-27 (v0.44.6.20050527)
 *  - added NV|U and netscape 8. bumped maxVersion for DeerPark Alpha
 *
 * modified (Avi Halachmi): 2007-10-26 (v0.44.10.20071026)
 *  - added an option to capture keyboard up/down as smoothwheel events
 *
 * modified (Avi Halachmi): 2007-11-22 (v0.44.11.2007122)
 *  - update url is now https for firefox 3 compatibility
 *
 * modified (Avi Halachmi): 2008-05-07 (v0.44.12.20080507)
 *  - update url is back to http, added signature (downloads.mozdev.org doesn't support https well - certs)
 *  - added horizontal scroll optional fix
 *
 * modified (Avi Halachmi): 2008-05-09 (v0.44.14.20080509)
 *  - Fix: smoothwheel now doesn't intercept events it shouldn't (i.e. if CTRL is held and s.w. isn't configured to use it). bug was due to behaviour change on firefox 3.
 *  - META modifier is now supported (OS/X)
 *
 * modified (Avi Halachmi): 2008-10-03 (v0.44.16.20081003)
 *  - Improvement: Soft edge was slowing down too much towards the edge if kept scrolling the wheel
 *       (was "extending" the deadline on repeated wheel events). Fixed.
 *  - Fix horizontal scroll break, Only in FX 3.1+ gecko engine (see https://bugzilla.mozilla.org/show_bug.cgi?id=378028 )
 *
 * modified (Avi Halachmi): 2009-03-31 (v0.44.17.20090331)
 *  - Fix: Flock v2+ : options panel was messed up
 *  - Fix: Firefox v3.6+ (trunk): Gecko 1.9.2+ doesn't support getBoxObjectFor for html documents
 *  - Added donations panel on options
 *
 * modified (Avi Halachmi): 2009-04-08 (v0.44.18.20090408)
 *  - Fix: Keep preferences on upgrade if version is compatible
 *  - Fix: Compliance with new AMO requirements (JS closure)
 * 
 * modified (Avi Halachmi): 2009-08-09 (v0.44.19.200900811)
 *  - New: Icon
 *  - Improve: modifier on/off affects also while scrolling
 *  - Improve: slightly wider range of step sizes
 *  - Compatibility: Firefox trunk (3.6a1pre) - added manifest
 *  - remarked: New: System/Global acceleration option (advanced -> Adaptive-Step drop down)
 *  - remarked: Improve: Better step accuracy on fast wheel scroll
 *
 * modified (Avi Halachmi): 2010-01-10 (v0.45.4.20100110)
 *  - Improve: Target detection - rewrite of detection code, adding many refinements and improvements, for both wheel and KB events
 *    - Examples of previously broken: print-preview, calcalist.co.il, wikipedia at the bottom, some boxes on ynet and more
 *    - Reverts to default scroll: embeded elements (flash, video, etc), over a closed and active drop-down
 *    - All else that I've tested (many) seem 100% OK
 *  - Change: Removed "Compatibility mode"
 *  - New: Keyboard shortcut for "quick disable", configured from the advanced panel (disabled by default)
 *  - Improved: SongBird support (actual scroll steps were longer than intended because events were handled twice)
 *  - Refined texts on config panels and better help tab
 *  - Code cleanups (remarks, unused code, indentations)
 *  - Refresh rate: More options (240,500,1000hz, as-fast-as-possible), Better timing (automatically use nsITimer if available)
 *  - New: fun statistics tab on the config dialog
 *  - New: landing page
 *
 * modified (Avi Halachmi): 2010-01-30 (v0.45.5.20100130)
 *  - Fixed regression: was jumping if scroll started really fast (flick of the wheel or some touchpads)
 *  - Improved: better scroll when using full page zoom
 *  - New: 2 more speed options: 0.4s, 0.7s. Default is now 0.4s (was 0.3).
 *  - New: optional log into a single pref value (set logDebug=5, logs to recent_log). log counter decreases automatically on each scroll stop, untill disabled.
 *  - Compliance: less use of exceptions, few objects cleanups when scroll ends
 *
 * modified (Avi Halachmi): 2010-02-02 (v0.45.6.20100202)
 *  - Fixed regression of the regression fix: Page would automatically jump to the left if horiz scrollbar was not leftmost
 *********************************************************/

if(!org) var org={};
if(!org.mozdev) org.mozdev={};
if(!org.mozdev.smoothwheel) org.mozdev.smoothwheel={};

org.mozdev.smoothwheel.ext={

//**************** CONFIGURATION SECTION ****************

// ********** IMPORTANT ***************
//    Mozilla/Netscape users, enter this address: chrome://smoothwheel/content/prefoverlay.xul
///////////////////////////////////////

sw_package:"AMO",
//sw_package:"mozdev.org",
sw_currentVersion: "0.45.6.20100202",

sw_MinConfigVersion: "0.45.1", //if the version on preferences is lower than this one, wipe preferences
//0.45.1.2: changed fps mode
//0.44.16: added prefs

sw_minPopupVersion: "0.45.4",
//0.45.4: added vsync tip



//Global enable/disable for this extension. default is true.
// setting it to false will disable this extension completely.
sw_EnableThisExtension: true,

//max milisec to complete scroll after last scroll event
//reasonable values are (fast)100/250/500/1000(slow). default is 300
// -> this value is overriden if a preset is selected.
sw_ScrollDuration: 300,

//if enabled, set step size relative to the scrolled area height by a constant factor.
//thus, if enabled and the visible window/frame is short, so will be the scroll steps.
//when enabled, it overrides gecko internal fixed pixels/line constant. default is true.
// -> this value is overriden if a preset is selected.
sw_EnableRelativeStepSize: true,

//if enabled, the step size will vary according to the delay between consecutive wheel events.
//such that fast wheel scroll will cause step size to increase, and vice versa.
//the desired effect is that when you're reading, and want to scroll, the step is small,
//but it you're just browsing through the document, with faster wheel action,
//you'll gain more 'milage' per wheel event. default is true.
// -> this value is overriden if a preset is selected.
sw_EnableAdaptiveStep: true,

//if enabled, duration will vary according to the delay between consecutive wheel events,
//such that fast wheel scroll will cause duration to DEcrease, and vice versa.
//the desired effect is to make it slower and smoother while reading, but 'snappy' when browsing.
//default is true.
// -> this value is overriden if a preset is selected.
sw_EnableAdaptiveDuration: true,

//if enabled, causes the scroll to 'land gracefully' on the top or bottom edges of the document.
//default is true.
sw_EnableSoftEdge: true,

//Limits the frames/sec of the smooth scroll operation.
//for best visual performance, set this value to your monitor refresh rate. default is 100.
//reasonable values are 20 to 120
//note that setting it to high values will make no difference if your cpu isn't up for the task.
//tips: for faster Gecko rendering, set Desktop color to 16bpp, and turn off font-anti-aliasing
//IMPORTANT TIP: if possible, set your monitor refresh rate to 100, and ScrollMaxFPS to either 100 or 50.
//   (other values will cause imprecision due to low resolution internal timers)
//2010-01: info: it appears that up to about 70fps the timer works reasonably well. faster than that, and it doesn't
//go any quicker. So, for as fast as possible (seem to be about 95fps with loop via setTimeout(func,0), even if page is practically empty
//  set this val to 0 (as an indication, isn't treated as actual fps value)
//2010-01-22//reserving 1 also. actual fps mode is 0=brute force one shots, else repeater timer, default is 240 hz
sw_ScrollMaxFPS: 240,

//IMPORTANT NOTE ABOUT MODIFIER KEYS:
//by default, mozilla/phoenix/firebird assigns the ctrl key to enlarge/reduce the text size.
//the alt key is assigned to browse history. shift key is not assigned.
//in order for a modifier key to work with this extension, it has to be UNassigned first.
//you can UNassign a modifier key for the mouse wheel as follows:
//mozilla: edit->preferences->advanced->mouse Wheel -> set it to scroll normally.
//phoenix/firebird: about:config in the locationbar bar. then change mousewheel.with<xxx>key.action property to 0

//Select a modifier key for fine small scroll steps. default is 3 (SHIFT key).
//possible values: 0=Disable, 1=ALT, 2=CTRL, 3=SHIFT, 4=META
sw_ModifierSmallStep: 3,

//Select a modifier key for full page scroll (like PgUp/PgDn). default is 1 (ALT key).
//possible values: 0=Disable, 1=ALT, 2=CTRL, 3=SHIFT, 4=META
sw_ModifierBigStep: 1,

//Enable/Disable capture supported keyboard events for scroll (Up/Down arrows)
sw_enableKeyboard: false,


//********* END OF COMMON CONFIGURATION OPTIONS SECTION ***********

//WHILE THE ABOVE COMMON CONFIGURATIONS ARE MOSTLY ENABLE/DISABLE, THE NEXT
//VALUES CAN BE USED TO FINE-TUNE THE BEHAVIOUR OF EACH FEATURE

//these 2 values control the step size when EnableAdaptiveStep=true.
//they define the factors for the smallest and largest steps, respectively. defaults are .4 and 1.2
//for extreme control (and variation) (requires a steady hand as well), try 0.3 and 3 respectively.
sw_AdaptiveStepMinStepFactor: 0.4,
sw_AdaptiveStepMaxStepFactor: 1.2,

//these 2 values control the duration when EnableAdaptiveDuration=true.
//they define the factors for the smallest and largest durations, respectively. defaults are .8 and 2
sw_AdaptiveDurationMinStepFactor: 0.8,
sw_AdaptiveDurationMaxStepFactor: 2,

//factor to use when EnableRelativeStepSize is true. default is 0.15 of viewable page height.
sw_RelativeStepFactor: 0.15,

//this factor configures how much "big step" is when ModifierBigStep is used.
//1 will result in normal step. 2 results 2 steps, etc.
//default is 5 of full page height.
sw_BigStepFactor: 5,

//factor for how short the scroll steps are if ModifierSmallStep is used.
//default is 0.1 of normal step.
sw_ShortStepFactor: 0.2,

//#pixels per scroll line. relevant only if EnableRelativeStepSize=false.
//default value is 20 (empirical estimation of internal Gecko value)
sw_ScrollPx: 20,

//the next 4 values control the duration between wheel events that define the
//adaptive behaviour of the adaptiveStep and adaptiveDurations features.

//defaults are .7 and 1.5
sw_AdaptiveStepMinDurationFactor: 0.7,
sw_AdaptiveStepMaxDurationFactor: 1.5,

//defaults are .7 and 1.5
sw_AdaptiveDurationMinDurationFactor: 0.7,
sw_AdaptiveDurationMaxDurationFactor: 1.5,

//******* it's recommended not to change the next values since it'll prevent optimizations *******/

//controls debug level. 3 is maximum, default is 0.
sw_debugMode: 0,

//keyboard toggle of smoothwheel, key code (CTRL+SHIFT should also be pressed). 0 --> no KB toggle
sw_kbToggle_DOM_VK: 0,
//*************** END OF CONFIGURATION SECTION ********************

sw_LoopInterval: 0, //fps converted to ms between scroll events (calculated on initGlobals)

//Global variables (used by more than one function).
sw_scrollLeft: 0, //in pixels
sw_scrollLeftLastEvent: 0, //in pixels
sw_scrollEventTimeStamp: 0,
sw_scrollDeadline: 0,// timestamp
sw_clientFrame: null,  //holds the object to scroll
sw_intervalID: 0,   //holds the interval timer handle. also indicates whether we're currently scrolling.
sw_timeoutID: 0,
sw_velocityLastEvent: 0,
sw_velocityCurrent: 0,

sw_scrollCurrentDuration: 0,  //how many ms to assign for the current scroll event.

//set some of the adaptive functionality parameters only once in advance.
sw_ASa: 0,
sw_ASb: 0,
sw_ADa: 0,
sw_ADb: 0,

//some collected statistics
sw_stat_handledEvents:0,
sw_stat_seconds:0,
sw_stat_pages:0,


//user preference: not read into a variable but rather directly set sw_timerMode according to its value and sw_canUse_nsITimer_result
//useLegacyTiming

//result of testing wheather or not nsITimer is available on this platform.
sw_canUse_nsITimer_result:false,

//actual usage in smoothwheel: 0: setTimeout/setInterval, 1: nsITimer
sw_timerMode:0,



sw_exceptionDbg: function(text, e){
    if (undefined==e) e="[not-provided]"; if (undefined==text) text="[not-provided]";
    var sw=org.mozdev.smoothwheel.ext;
    sw.sw_debugMode&&sw.dbg("!! ---> Exception ("+text+") : "+e);
},

sw_initGlobals: function(){with (org.mozdev.smoothwheel.ext){
    sw_stopScroll();

    sw_LoopInterval=1;
    if (sw_ScrollMaxFPS>0) sw_LoopInterval=Math.round(1000/sw_ScrollMaxFPS);

    sw_UAafterRV=window.navigator.userAgent.lastIndexOf("rv:") + 3;
    sw_geckoVer=parseFloat(window.navigator.userAgent.substring(sw_UAafterRV));
    sw_geckoPre_1_7=((isNaN(sw_geckoVer) || sw_geckoVer>=1.7)?false:true);


    sw_ASa=((sw_AdaptiveStepMinStepFactor-sw_AdaptiveStepMaxStepFactor)/
          (sw_AdaptiveStepMaxDurationFactor-sw_AdaptiveStepMinDurationFactor))/sw_ScrollDuration;//diagonal
    sw_ASb=sw_AdaptiveStepMaxStepFactor-sw_ASa*sw_AdaptiveStepMinDurationFactor; //offset

    sw_ADa=((sw_AdaptiveDurationMaxStepFactor-sw_AdaptiveDurationMinStepFactor)/
          (sw_AdaptiveDurationMaxDurationFactor-sw_AdaptiveDurationMinDurationFactor))/sw_ScrollDuration;//diagonal
    sw_ADb=sw_AdaptiveDurationMinStepFactor-sw_ADa*sw_AdaptiveDurationMinDurationFactor; //offset

    sw_scrollLeft=0;
    sw_scrollLeftLastEvent=0;
    sw_scrollEventTimeStamp=0;
    sw_scrollDeadline=0;
    sw_clientFrame=null;
    sw_intervalID=0;
    sw_timeoutID=0;
    sw_velocityLastEvent=0;
    sw_velocityCurrent=0;
    sw_scrollCurrentDuration=0;
    
    sw_stat_handledEvents=0;
    sw_stat_seconds=0;
    sw_stat_pages=0;

    return;
}},

sw_dbgAlert: function(x){if (org.mozdev.smoothwheel.ext.sw_debugMode>0) return alert(x);},
sw_dbgStringObj: function(x){
    try{
        return ""+x+"='"+org.mozdev.smoothwheel.ext[x]+"'\n";
    }catch(e){
        return ""+x+"=[* Unable to evaluate! *]\n";
    }
},
//if some sanity tests fail for no appearant reason, reset to initial values.
sw_panicAbort: function(){with (org.mozdev.smoothwheel.ext){
  sw_dbgAlert("SmoothWheel: panic abort:\nPlease post the current page on the smoothwheel forum\n\n"+
    sw_dbgStringObj("sw_scrollLeft")+
    sw_dbgStringObj("sw_scrollLeftLastEvent")+
    sw_dbgStringObj("sw_scrollEventTimeStamp")+
    sw_dbgStringObj("sw_scrollDeadline")+
    sw_dbgStringObj("sw_clientFrame")+
    sw_dbgStringObj("sw_intervalID")+
    sw_dbgStringObj("sw_velocityLastEvent")+
    sw_dbgStringObj("sw_velocityCurrent")+
    sw_dbgStringObj("sw_scrollCurrentDuration")+
    "");

  sw_initSmoothWheel();
}},

sw_prefServices: Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefBranch),

sw_setPref: function(aPrefType, aPrefString, aValue){with (org.mozdev.smoothwheel.ext){
//    sw_debugMode&&dbg("set pref ("+aPrefType + ", "+aPrefString+", "+aDefaultValue+")");

    try {
      switch (aPrefType) {
        case "bool":
          sw_prefServices.setBoolPref(aPrefString, aValue);
          break;
        case "int":
          sw_prefServices.setIntPref(aPrefString, aValue);
          break;
        case "color":
        case "string":
        case "localizedstring":
        default:
        sw_prefServices.setCharPref(aPrefString, aValue);
        break;
      }
    }catch(e){
//      if(_DEBUG) dump(e + "\n");
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_setPref", e);
    }
}},

sw_PrefExtensionRoot: "extensions.smoothwheel.",
sw_PrefExtensionStatsRoot: "extensions.smoothwheelStatistics.",

sw_getPrefOpts: function(aPrefType, aPrefString, aDefaultValue, isCreateOnNone){with (org.mozdev.smoothwheel.ext){
//    sw_debugMode&&dbg("get pref ("+aPrefType + ", "+aPrefString+", "+aDefaultValue+")");
//    if(_DEBUG) dump("in getpref\n");
    try {
      switch(aPrefType) {
        case "bool":
          return sw_prefServices.getBoolPref(aPrefString);
          //return !aDefaultFlag ? pref.getBoolPref(aPrefString) : pref.getDefaultBoolPref(aPrefString);
        case "int":
          return sw_prefServices.getIntPref(aPrefString);
          //return !aDefaultFlag ? pref.getIntPref(aPrefString) : pref.getDefaultIntPref(aPrefString);
        case "localizedstring":
          return sw_prefServices.getLocalizedUnicharPref(aPrefString); // not working?
        case "color":
        case "string":
        default:
          return sw_prefServices.getCharPref(aPrefString); 
          //return !aDefaultFlag ? pref.copyUnicharPref(aPrefString) : pref.copyDefaultUnicharPref(aPrefString);
      }
    }catch(e) {
       //~ if(_DEBUG) {
        //~ dump("*** no default pref for " + aPrefType + " pref: " + aPrefString + "\n");
        //~ dump(e + "\n");
      //~ }
//      alert ("get pref ("+aPrefString + ", "+aPrefType+") failed");
//      sw_prefServices.deleteBranch(aPrefString);
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_getPrefOpts", e);
        if (isCreateOnNone)
            sw_setPref(aPrefType, aPrefString, aDefaultValue);
      
        return aDefaultValue;
    }
    return "!/!ERROR_UNDEFINED_PREF!/!";
}},

sw_getPref: function(aPrefType, aPrefString, aDefaultValue, isCreateOnNone){
    if (undefined==isCreateOnNone) isCreateOnNone=true;
    return org.mozdev.smoothwheel.ext.sw_getPrefOpts(aPrefType, aPrefString, aDefaultValue, isCreateOnNone);
},


//returns -1 if v1<v2, 0 if equal and 1 if v1>v2, where v1 and v2 are "xx.yy.zz...".
//if a value cannot be parsed, it's considered as 0
//if non-numeric chars are present in either value, result is undefined (i.e. possibly not what's expected)
sw_CompareVersions: function(v1, v2){
    var v1f=parseFloat(v1); if (isNaN(v1f)) v1f=0;
    var v2f=parseFloat(v2); if (isNaN(v2f)) v2f=0;

    if (v1f<v2f) return -1;
    if (v1f>v2f) return 1;
    if (v1.indexOf(".")<0 && v2.indexOf(".")<0) return 0;
    
    if (v1.indexOf(".")<0) v1+=".0";
    if (v2.indexOf(".")<0) v2+=".0";
    
    return org.mozdev.smoothwheel.ext.sw_CompareVersions(v1.substring(v1.indexOf(".")+1), v2.substring(v2.indexOf(".")+1));
},


//From Foxy-Proxy version 2.18 (File:common.js, License: GPL V2 which is same as SmoothWheel's license): start
//Foxy-Proxy's code is based on the snippest here: https://developer.mozilla.org/En/Code_snippets/Tabbed_browser
//the next 2 functions are used to open a new URL in the most appropriate window.
getMostRecentWindow: function(wm){
    const CI = Components.interfaces;
    const CC = Components.classes;
    var tmp = wm || CC["@mozilla.org/appshell/window-mediator;1"].getService(CI.nsIWindowMediator);
    return tmp.getMostRecentWindow("navigator:browser") || tmp.getMostRecentWindow("Songbird:Main") || tmp.getMostRecentWindow("mail:3pane");
},
// Application-independent version of getEnumerator()
getEnumerator: function() {
    const CI = Components.interfaces;
    const CC = Components.classes;
    var wm = CC["@mozilla.org/appshell/window-mediator;1"].getService(CI.nsIWindowMediator);
    // The next line always returns an object, even if the enum has no elements, so we can't use it to determine between applications
    //var e = wm.getEnumerator("navigator:browser") || wm.getEnumerator("Songbird:Main")
    return wm.getMostRecentWindow("navigator:browser") ? wm.getEnumerator("navigator:browser") : (wm.getEnumerator("Songbird:Main") ? wm.getEnumerator("Songbird:Main") : wm.getEnumerator("mail:3pane"));
},
openAndReuseOneTabPerURL: function(aURL, referrerURL){
    if (undefined==referrerURL) referrerURL=null;
    try{
        const CI = Components.interfaces;
        const CC = Components.classes;
      
        var wm = Components.classes["@mozilla.org/appshell/window-mediator;1"].getService(CI.nsIWindowMediator);
        var winEnum = wm.getEnumerator("navigator:browser");
        if (!winEnum.hasMoreElements())
            winEnum = wm.getEnumerator("Songbird:Main");
        if (!winEnum.hasMoreElements())
            winEnm = wm.getEnumerator("mail:3pane");
        while (winEnum.hasMoreElements()){
            var win = winEnum.getNext();
            var browser = win.getBrowser();
            for (var i = 0; i < browser.mTabs.length; i++) {
                if (aURL == browser.getBrowserForTab(browser.mTabs[i]).currentURI.spec) {
                  win.focus(); // bring wnd to the foreground
                  browser.selectedTab = browser.mTabs[i];
                  return;
                }
            }
        }

        // Our URL isn't open. Open it now.
        var w = this.getMostRecentWindow(wm);
        var event = { notify: function(timer) {w.gBrowser.selectedTab = w.gBrowser.addTab(aURL, referrerURL, null, null);} }

        if (w){
          // Note: Since TB doesn't support tabs and trunk isn't the same
          // Use an existing browser window
            if(w.messenger) // Thunderbird
                w.messenger.launchExternalURL(aURL);
            else if (!w.delayedOpenTab) /* SongBird, etc. */{
                //setTimeout(function(aTabElt) { w.gBrowser.selectedTab = aTabElt; }, 0, w.gBrowser.addTab(aURL, null, null, null));
                var t = Components.classes["@mozilla.org/timer;1"].createInstance(CI.nsITimer);
                t.initWithCallback(event, 10, CI.nsITimer.TYPE_ONE_SHOT);
            }
            else // FF, SM, Flock, etc.
                w.delayedOpenTab(aURL, referrerURL, null, null, null);
                
            w.focus();
        }
    } catch(e){
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("openAndReuseOneTabPerURL", e);
    }
},
//from Foxy-Proxy: end



sw_openBrowserWindow: function(url, referrer){with (org.mozdev.smoothwheel.ext){
    try{
        openAndReuseOneTabPerURL(url, referrer);
        return;
    } catch(e){
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_openBrowserWindow", e);
    }
}},

sw_getWelcomePageUrl: function(isNewInstall){with (org.mozdev.smoothwheel.ext){
    return "http://smoothwheel.mozdev.org/"+(isNewInstall?"n":"u")+(sw_package=="AMO"?"a":"m")+"_welcome.php";
}},

sw_needsVisit:"",
sw_logDebug:0,

sw_readPrefs: function(){with (org.mozdev.smoothwheel.ext){
    //1st, cleanup pref data if doesn't belong to the current smoothwheel version.
    try {
    
        var prefVersion=sw_prefServices.getCharPref(sw_PrefExtensionRoot+"version");
        if (prefVersion != sw_currentVersion){
            if (sw_CompareVersions(prefVersion, sw_MinConfigVersion) <0 || sw_CompareVersions(prefVersion, sw_currentVersion)>0){
                //if pref version lower than minimum or if pref version higher than current version (going to earlier version), wipe prefs
                sw_prefServices.deleteBranch(sw_PrefExtensionRoot);
                //sw_debugMode&&dbg("deleting SW prefs\ncurrent="+sw_currentVersion+"\nmin="+sw_MinConfigVersion+"\nprefv="+prefVersion);
            } else{
                //alert("keeping SW prefs\ncurrent="+sw_currentVersion+"\nmin="+sw_MinConfigVersion+"\nprefv="+prefVersion);
            }
            
            if (sw_CompareVersions(prefVersion, sw_minPopupVersion)<0){
                sw_prefServices.setCharPref(sw_PrefExtensionRoot+"needs_visit",sw_getWelcomePageUrl(false));
            }
                
            sw_prefServices.setCharPref(sw_PrefExtensionRoot+"version", sw_currentVersion);
        }
    } catch (e){//if value isn't in prefs
        sw_prefServices.deleteBranch(sw_PrefExtensionRoot);
        sw_prefServices.setCharPref(sw_PrefExtensionRoot+"version", sw_currentVersion);
        sw_prefServices.setCharPref(sw_PrefExtensionRoot+"needs_visit",sw_getWelcomePageUrl(true));   //new user
    }
    sw_EnableThisExtension          = !sw_getPref("bool",       sw_PrefExtensionRoot+ "quickDisable", false);
    sw_kbToggle_DOM_VK              = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "kbToggle_DOM_VK", "0");
    if (!sw_EnableThisExtension)
        return;
        
    sw_needsVisit=""+sw_getPref("string",    sw_PrefExtensionRoot+ "needs_visit", "");
    
    sw_ScrollDuration               = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "scrollDuration", "400");

    var css=1*sw_getPref("string",    sw_PrefExtensionRoot+ "compoundStepSize", "0.2");
    if (css>0){
        sw_EnableRelativeStepSize=true;
        sw_RelativeStepFactor=css;
    } else{
        sw_EnableRelativeStepSize=false;
        sw_ScrollPx=-css;
    }

    sw_ModifierBigStep              = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "fastScrollModifier", "1");
    sw_ModifierSmallStep            = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "slowScrollModifier", "3");

    var pas=1*sw_getPref("string",    sw_PrefExtensionRoot+ "presetAdaptiveStep", "2");
    sw_EnableAdaptiveStep=true;
    switch (pas){
        case 0: sw_EnableAdaptiveStep=false; break;
        case 1:sw_AdaptiveStepMinStepFactor=0.6;sw_AdaptiveStepMaxStepFactor=1.1;break;
        case 2:sw_AdaptiveStepMinStepFactor=0.4;sw_AdaptiveStepMaxStepFactor=1.2;break;
        case 3:sw_AdaptiveStepMinStepFactor=0.3;sw_AdaptiveStepMaxStepFactor=1.5;break;
        case 4:sw_AdaptiveStepMinStepFactor=0.2;sw_AdaptiveStepMaxStepFactor=2;break;
        default:sw_AdaptiveStepMinStepFactor=0.4;sw_AdaptiveStepMaxStepFactor=1.2;break;
    }

    var pad=1*sw_getPref("string",    sw_PrefExtensionRoot+ "presetAdaptiveDuration", "2");
    sw_EnableAdaptiveDuration=true;
    switch (pad){
        case 0: sw_EnableAdaptiveDuration=false; break;
        case 1: sw_AdaptiveDurationMinStepFactor=0.9;sw_AdaptiveDurationMaxStepFactor=1.5;break;
        case 2: sw_AdaptiveDurationMinStepFactor=0.8;sw_AdaptiveDurationMaxStepFactor=2;break;
        case 3: sw_AdaptiveDurationMinStepFactor=0.5;sw_AdaptiveDurationMaxStepFactor=2.5;break;
        case 4: sw_AdaptiveDurationMinStepFactor=0.3;sw_AdaptiveDurationMaxStepFactor=3;break;
        default:sw_AdaptiveDurationMinStepFactor=0.8;sw_AdaptiveDurationMaxStepFactor=2;break;
    }

    sw_ScrollMaxFPS                 = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "scrollMaxFPS", "240");
    sw_LoopInterval=1;
    if (sw_ScrollMaxFPS>0) sw_LoopInterval=Math.round(1000/sw_ScrollMaxFPS);
    sw_EnableSoftEdge               = sw_getPref("bool",        sw_PrefExtensionRoot+ "enableSoftEdge", true);
    sw_ShortStepFactor              = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "shortStepFactor", "0.2");
    sw_BigStepFactor                = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "bigStepFactor", "5");
    sw_enableKeyboard               = sw_getPref("bool",        sw_PrefExtensionRoot+ "enableKeyboard", false);
    sw_debugMode                    = 1*sw_getPref("int",       sw_PrefExtensionRoot+ "debugMode", 0);
    sw_logDebug                     = 1*sw_getPref("string",    sw_PrefExtensionRoot+ "logDebug", 0);
    if (sw_logDebug) sw_debugMode=3;


    var timingPref=sw_getPref("bool", sw_PrefExtensionRoot+ "useLegacyTiming", false)?0:1;
    sw_timerMode=sw_canUse_nsITimer_result?timingPref:0;
    
//    dbg("sw_readPrefs: end");
}},


sw_logCurrent:"",
sw_logPrevious:"",
sw_updateLog: function(){with (org.mozdev.smoothwheel.ext){
    if (!sw_logDebug)
        return;
    
    sw_logDebug--;
    sw_setPref("string",   sw_PrefExtensionRoot+ "logDebug", sw_logDebug);
    
    if (sw_logDebug<=0){
        //automatically turn off logging
        sw_logDebug=0;
        sw_debugMode= 1*sw_getPref("int",       sw_PrefExtensionRoot+ "debugMode", 0);
        sw_setPref("string", sw_PrefExtensionRoot+ "logDebug", 0);
        sw_setPref("string", sw_PrefExtensionRoot+ "recent_log", "");
        sw_logPrevious=sw_logCurrent="";
        return;
    }

    //implicit else: update log
    sw_setPref("string", sw_PrefExtensionRoot+ "recent_log", sw_logPrevious+sw_logCurrent);
    sw_logPrevious=sw_logCurrent; sw_logCurrent="";
}},

sw_saveStats: function(){with (org.mozdev.smoothwheel.ext){
    sw_clientFrame=null;
    if (sw_stat_seconds<=0 || sw_stat_seconds>30){
        sw_debugMode&&dbg("!!Stats not available or invalid. Reset temp stat and abort save");
        sw_stat_handledEvents=sw_stat_seconds=sw_stat_pages=0;
        sw_logDebug&&sw_updateLog();
        return;
    }
    
    function secsToReadableDuration(s, optionalMaxElements){
        if (undefined==optionalMaxElements) optionalMaxElements=2;
        function toNSP(num, single){
            if (num==0) return "";
            return ""+num+" "+single+(num==1?"":"s");
        }
        
        s=Math.round(s);
        var secs=s%60, mins=Math.floor(s/60)%60, hours=Math.floor(s/3600)%24, days=Math.floor(s/86400);
        var res="", tmp="", c=0, i=0;
        var info=[[days,"day"], [hours,"hour"], [mins,"minute"], [secs,"second"]];
        
        for (i=0; i<info.length; i++){
            tmp=toNSP(info[i][0], info[i][1]); if (tmp!="" || c>0) c++;
            if (res!="" && tmp!="") res+=", ";
            res+=tmp;
            if (c>=optionalMaxElements)
                return res;
        }
        if (res=="") res="0 seconds"; 
        return res;
    };
    
    //sw_debugMode&&dbg("!SmoothWheel: Starting statistics update");

    var prefEvents=         1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_handledEvents", "0");
    var presSeconds=        1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_seconds", "0");
    var prefPages=          1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_pages", "0");
    var prefDaysUsed=       1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_daysUsed", "0");
    var prefLastEventDate=  1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_lastEventDate", "0");
    var prefCollectedSince= 1*sw_getPref("string",    sw_PrefExtensionStatsRoot+ "stat_collectedSince", "0");

    //sw_debugMode&&dbg("!Previous stats: since: "+prefCollectedSince+", days used: "+prefDaysUsed+", events handled: "+prefEvents+", total seconds: "+presSeconds+", pages: "+prefPages);
   
   
    prefEvents  += sw_stat_handledEvents ;
    presSeconds += sw_stat_seconds;
    prefPages   += sw_stat_pages;
    try{sw_debugMode&&dbg("!Stats: events+"+sw_stat_handledEvents+", secs+"+sw_stat_seconds+", pages+"+sw_stat_pages);} catch(e){};

    sw_stat_handledEvents=sw_stat_seconds=sw_stat_pages=0;

    var today=Math.floor(((new Date()).getTime())/86400000);
    if (today!=prefLastEventDate)
        prefDaysUsed++;
    prefLastEventDate=today;
    if (prefCollectedSince==0)
        prefCollectedSince=today;
        
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_handledEvents",   prefEvents);
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_seconds",         presSeconds);
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_pages",           prefPages);
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_daysUsed",        prefDaysUsed);
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_lastEventDate",   prefLastEventDate);
    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_collectedSince",  prefCollectedSince);
    
    var sb="<html:b>", eb="</html:b>";
    var disp=""
    try{
        disp+=  
            "<html:ul><html:li>Since "+sb+(new Date(prefCollectedSince*86400000)).toLocaleDateString()+eb+", SmoothWheel was used on "+
            prefDaysUsed+" days ("+Math.round(100*prefDaysUsed/(1+prefLastEventDate-prefCollectedSince))+"% of the days),"+
            " rolled about "+sb+Math.round(prefPages)+" full pages"+eb+" and handled "+sb+prefEvents+" scroll events"+eb+
            ". Combined, it was scrolling for "+sb+secsToReadableDuration(presSeconds,4)+eb+"!"+"</html:li>"+
            "<html:li>"+sb+"On average"+eb+", SmoothWheel was invoked "+
            sb+Math.round(prefEvents/prefDaysUsed)+" times a day"+eb+" and was scrolling for over "+
            sb+secsToReadableDuration(presSeconds/prefDaysUsed,1)+" a day"+eb+".</html:li></html:ul>";

    }catch(e){
        disp="[ Unable to display statistics at the moment, please report this bug ]";
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("stats display text", e);
    }

    sw_setPref("string", sw_PrefExtensionStatsRoot+ "stat_display", disp);
    //sw_debugMode&&dbg("!Stats: since: "+prefCollectedSince+", days used: "+prefDaysUsed+", events handled: "+prefEvents+", total seconds: "+presSeconds+", pages: "+prefPages);
    sw_logDebug&&sw_updateLog();
}},


//checks whether the specified modifier key was held when the event was intercepted.
sw_modifierHeld: function(e, modifier){
    switch (modifier) {
       case 1: return e.altKey;
       case 2: return e.ctrlKey;
       case 3: return e.shiftKey;
       case 4: return e.metaKey;
    }
    return false;
},

//checks wheather a modified is held which is not configured to be used in smoothWheel
//i.e. CTRL is used to resize text by default. If smoothwheel isn't configured to use CTRL but it's held,
//then smoothwheel shouldn't handle it.
//this is due to behaviour change in firefox 3 itself.
// in firefox <= 2.0.* the event wasn't passed to smoothwheel if it was used by firefox.
// on firefox 3 it's 1st passed to other handlers, including smoothwheel.
sw_isUnconfiguredModifierHeld: function(e){with (org.mozdev.smoothwheel.ext){
    if (   0
        || e.altKey   && sw_ModifierBigStep!=1 && sw_ModifierSmallStep!=1
        || e.ctrlKey  && sw_ModifierBigStep!=2 && sw_ModifierSmallStep!=2
        || e.shiftKey && sw_ModifierBigStep!=3 && sw_ModifierSmallStep!=3
        || e.metaKey  && sw_ModifierBigStep!=4 && sw_ModifierSmallStep!=4
        ){
        return true;
    }
    
    return false;
}},


sw_isInParentsTree: function(testNode, possibleParentNode, isWindow){
    try{
        if (!isWindow) return true;
        
        //else jump windows up, try to find it.                
        var win;
        if (testNode.ownerDocument && (win=testNode.ownerDocument.defaultView)){
            if (possibleParentNode==win)
                return true;

            if (win.frameElement && (win=win.frameElement.parentNode))//recurse upper win if exists
                return org.mozdev.smoothwheel.ext.sw_isInParentsTree(win, possibleParentNode, isWindow);
        }
        
        var sw=org.mozdev.smoothwheel.ext;
        sw.sw_debugMode&&sw.dbg("!Not same win");
        return false;//top win and not found

    }catch(e){
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("at sw_isInParentsTree", e);
    }
    //sw.sw_debugMode&&sw.dbg("!! exception (check win).");
    return true;//previous implementation was always true. keep it if we're lost.
},



//detect new (gecko 1.7 and above cvs trees) browsers
sw_UAafterRV: 0,//window.navigator.userAgent.lastIndexOf("rv:") + 3,
sw_geckoVer: 0,//parseFloat(window.navigator.userAgent.substring(org.mozdev.smoothwheel.ext.sw_UAafterRV)),
sw_geckoPre_1_7: 0,//((isNaN(org.mozdev.smoothwheel.ext.sw_geckoVer) || org.mozdev.smoothwheel.ext.sw_geckoVer>=1.7)?false:true),
//alert("UA:\n"+window.navigator.userAgent+"\ngecko ver: "+sw_geckoVer+"\nis pre1.7: "+sw_geckoPre_1_7);

sw_isWindow:false,
sw_findTargetObject: function(e){with (org.mozdev.smoothwheel.ext){
    sw_isWindow=false;
    //next code is based on All-In-One-Gestures v0.96.
    //It was vastly modified, enhanced and re-written by avih

    function sw_findNodeToScroll(initialNode, isKeyboard){
        function sw_aioGetStyle(elem, aProp){
          var p = elem.ownerDocument.defaultView.getComputedStyle(elem, "").getPropertyValue(aProp);
          var val = parseFloat(p);
          if (!isNaN(val)) return Math.ceil(val);
          if (p == "thin") return 1;
          if (p == "medium") return 3;
          if (p == "thick") return 5;
          return 0;
        }

        var realHeight, bodyEl, docEl, nextNode, currNode=null, prevNode, selFromOpt;
        docEl = initialNode.ownerDocument.documentElement;
        if (!docEl || docEl.nodeName.toLowerCase() != "html")
            return null;
        
        // walk the tree up looking for something to scroll
        var bodies = docEl.getElementsByTagName("body");
        if (!bodies || !bodies.length)
            return null;

        bodyEl = bodies[0];
        if (initialNode == docEl) nextNode = bodyEl;
        else nextNode = initialNode;
        do{
            try{
                //sw_debugMode&&dbg("!sw_findNodeToScroll: loop start: "+initialNode+", name="+initialNode.name+", type="+initialNode.type);
                prevNode = currNode; currNode = nextNode; name=currNode.nodeName.toLowerCase(); nextNode = currNode.parentNode; selFromOpt=false;

                if (isKeyboard&&
                        (""+currNode.contentEditable=='true' || name=="select" || name=="option" || name=="textarea"
                        || name=="input" && (currNode.type=="text" || currNode.type=="password" || currNode.type=="radio"))
                    )
                    return null;
                
                if (name=="body" || name=="html"){
                    try{
                        if (isKeyboard&&((''+currNode.ownerDocument.designMode).toLowerCase()=="on"))
                            return null;
                        
                        var win=currNode.ownerDocument.defaultView;
                        if (win.scrollbars.visible && win.scrollMaxY>0){
                            //sw_debugMode&&dbg("!Win: onscroll="+win.onScroll);
                            sw_isWindow=true;
                            return win;
                        }

                        if (win.frameElement && win.frameElement.parentNode)
                            return sw_findNodeToScroll(win.frameElement.parentNode);
                        return null;
                        
                    }catch (e){
                        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_findNodeToScroll-topmost", e);
                        return null;
                    }
                }
                
                var computedStyle=currNode.ownerDocument.defaultView.getComputedStyle(currNode, "");
                if (currNode.clientWidth && currNode.clientHeight && computedStyle.getPropertyValue("overflow") != "hidden" && computedStyle.getPropertyValue("overflow-y") != "hidden"){
                    
                    if (name=="embed")
                        return null;

                    if (name=="select"){
                        var ae=null;
                        try{ae=currNode.ownerDocument.activeElement;}catch(e){
                            org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_findNodeToScroll-select", e);
                        };
                        selFromOpt=prevNode&&prevNode.nodeName.toLowerCase()=="option";
                        
                        if (!ae||ae!=currNode)
                            continue;
                        if (!selFromOpt)
                            return null;
                    }

                    realHeight = currNode.clientHeight;
                    //avih: modified hack for FB 0.8+ MOZ 1.6+ etc from AIO v0.96
                    if (!sw_geckoPre_1_7 || currNode == bodyEl)
                        realHeight += sw_aioGetStyle(currNode, "border-top-width") + sw_aioGetStyle(currNode, "border-bottom-width");
                    
                    //sw_debugMode&&dbg("!node: "+currNode+", SH="+currNode.scrollHeight+", RH="+realHeight+", OH="+currNode.offsetHeight);
                    if (currNode.scrollHeight>realHeight && //standard test
                        realHeight*1.2>currNode.offsetHeight && (realHeight+40)>currNode.offsetHeight)//avih: ugly hack sanity
                        return currNode;
                }
            }catch(e){
                org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_findNodeToScroll-loop", e);
                return null;
            }
        } while (currNode && currNode != docEl && !selFromOpt);
        
        return null;
    }
    //avih: end of code originally based on AIO 0.96

    var scrollObj = sw_findNodeToScroll(e.target, (""+e.type).indexOf("key")==0);
    if (scrollObj && scrollObj.toString().indexOf("XULElement")<0)//not an XUL element
        return scrollObj;
    
    return false;
}},


sw_contains: function(a,b){
    return (""+a).indexOf(""+b)>=0;
},

sw_currentYPos:0,
sw_lastEstamp:0, sw_dtn_0: 0, sw_dtn_1: 0, sw_dtn_2: 0,
//determines whether to leave the event alone, or to use it (and initilize the vars)
sw_acquireEvent: function(e, isAlreadyScrolling){with (org.mozdev.smoothwheel.ext){
    
    //1. filter by event peroperties and known modifiers config
    if (sw_isUnconfiguredModifierHeld(e))
        return false;

    if (e.type=="keydown"){
        if (!sw_enableKeyboard || e.keyCode!=e.DOM_VK_DOWN && e.keyCode!=e.DOM_VK_UP)
            return false;
            
        e.sw_steps=(e.keyCode==e.DOM_VK_DOWN)?1:-1;
        
    } else{
        if (e.HORIZONTAL_AXIS!=undefined && e.axis==e.HORIZONTAL_AXIS)
            return false;

        //trying to normalize e.detail according to numlines, but inconsistent behavior on apps:
        // All apps don't expose sysnumlines (int), but events may accumulate (unable to conclude base num lines).
        // SM 1.1.7, 2b1: OK (sysnumlines ok, numlines ok, affecting)
        // FX 3.5.2: OK (sysnumlines ok, numlines ok, affecting)
        // FX 3.6a2pre: numlines: ok and affecting, sysnumlines acts as isUseAcceleration
        // -> conclusion: completely unreliable normalization, falling back to 1 event==1 step
        //      so instead of e.detail/sw_baseStepsInEvent I use sign(e.detail)
        e.sw_steps=(e.detail)>0?1:-1;//e.detail/sw_baseStepsInEvent
    }

    //2. initial filter passed. if not currently scrolling, search for scrollable node and abort if not found
    if (!isAlreadyScrolling){
        var tmpClientFrame= sw_findTargetObject(e);
        if (!tmpClientFrame)
            return false;

        //OK. intercepting event. Set global scrolled object
        sw_clientFrame= tmpClientFrame;
        if (sw_debugMode){
            var url=sw_isWindow?sw_clientFrame.location:sw_clientFrame.ownerDocument.defaultView.location; url=(""+url).split("?")[0].split("#")[0];
            var app=window.navigator?window.navigator.userAgent:"unknown";
            sw_debugMode&&dbg("!Scroll: type="+e.type+", found="+tmpClientFrame+" ("+e.target+"), pos="+sw_getScrollTop(sw_clientFrame)+"/"+sw_getPossibleScroll(sw_clientFrame)+"("+sw_getViewHeight(sw_clientFrame)+"), page='"+url+"', SW ver:"+sw_currentVersion+", App: "+app);
        }

        sw_currentYPos=sw_getScrollTop(sw_clientFrame);
    }

    //3. scrollable node known and passed modifiers test.
    //   Handle the event: initialize all required variables for this scroll
    var scrollCurrentStep=(sw_EnableRelativeStepSize)? sw_getViewHeight(sw_clientFrame)*sw_RelativeStepFactor: sw_ScrollPx;//absolute px amount
    
    if (sw_modifierHeld(e, sw_ModifierBigStep))
       scrollCurrentStep=scrollCurrentStep*sw_BigStepFactor;
    if (sw_modifierHeld(e, sw_ModifierSmallStep))
       scrollCurrentStep= scrollCurrentStep*sw_ShortStepFactor;
       
    var ets=e.timeStamp;
    if (ets===0)//some platforms
        ets=(new Date()).getTime();
    
    if (!isAlreadyScrolling){
        sw_dtn_1=sw_dtn_2=sw_ScrollDuration*sw_AdaptiveDurationMaxStepFactor;
    } else{
        sw_dtn_2=sw_dtn_1; sw_dtn_1=sw_dtn_0;
    }
    sw_dtn_0=ets-sw_lastEstamp;
    sw_lastEstamp=ets;
    var dtn=(sw_dtn_0*1+sw_dtn_1*1+sw_dtn_2)/3;

    var suggestedStepFactor=sw_clipValue(sw_ASa*dtn+sw_ASb, sw_AdaptiveStepMinStepFactor, sw_AdaptiveStepMaxStepFactor);
    var suggestedDurationFactor=sw_clipValue(sw_ADa*dtn+sw_ADb, sw_AdaptiveDurationMinStepFactor, sw_AdaptiveDurationMaxStepFactor);
    var suggested_scrollCurrentDuration=sw_ScrollDuration*((sw_EnableAdaptiveDuration)?suggestedDurationFactor:1);

    var scrollLeftBeforeAdjustment=sw_scrollLeft;
    sw_scrollLeft += e.sw_steps*scrollCurrentStep*(sw_EnableAdaptiveStep?suggestedStepFactor:1);

    if (sw_EnableSoftEdge){
        var destination=sw_getScrollTop(sw_clientFrame)+sw_scrollLeft;
        if (destination < 0) sw_scrollLeft = -sw_getScrollTop(sw_clientFrame);
        else if (destination > sw_getPossibleScroll(sw_clientFrame))
            sw_scrollLeft = sw_getPossibleScroll(sw_clientFrame) - sw_getScrollTop(sw_clientFrame);

        if (scrollLeftBeforeAdjustment==sw_scrollLeft){
            //no need to update deadline or anything (ignore event). already on path to soft edge with the same target
            sw_debugMode&&dbg("!event aborted on soft edge");
            return true;//no need to update any vars, but do not forfeit event, or else the default handler would kick in
        }
    }
    
    var now= (new Date()).getTime();
    sw_scrollCurrentDuration=suggested_scrollCurrentDuration;
    sw_scrollEventTimeStamp=sw_velocityCurrent?sw_lastWheelLoopEvent:now;
    var prevDL=sw_scrollDeadline;
    sw_scrollDeadline= sw_scrollEventTimeStamp + sw_scrollCurrentDuration;
    sw_scrollLeftLastEvent= sw_scrollLeft;
    sw_velocityLastEvent= sw_velocityCurrent;
    
    //start: statistics
    var secplus=(isAlreadyScrolling?(sw_scrollDeadline-prevDL):sw_scrollCurrentDuration)/1000;
    if (secplus>-10 && secplus<10){//sanity
        sw_stat_seconds += secplus;
        sw_stat_pages  += (Math.abs(scrollLeftBeforeAdjustment-sw_scrollLeft)/800);
        sw_stat_handledEvents++;
    }
    //end: statistics
    
    sw_debugMode&&dbg("!"+(isAlreadyScrolling?"Extending by ":"Setting to ")+Math.round(scrollLeftBeforeAdjustment-sw_scrollLeft)+"px, "+Math.round(isAlreadyScrolling?(sw_scrollDeadline-prevDL):sw_scrollCurrentDuration)+"ms");

    return true;
}},


sw_clipValue: function(value, minValue, maxValue){
  return (value>maxValue)?maxValue:((value<minValue)?minValue:value);
},


//utility funcs to get proper values wheather html object or [main] window
//shouldn't be called if target is invalid
/* weird, without this comments venkman identifies the next function as invalid...*/
sw_getScrollTop: function (item){
    if (org.mozdev.smoothwheel.ext.sw_isWindow) return item.scrollY;
    return item.scrollTop;
},
sw_getPossibleScroll: function (item){
    if (org.mozdev.smoothwheel.ext.sw_isWindow) return item.scrollMaxY;
    return item.scrollHeight-item.clientHeight;
},
sw_getViewHeight: function(item){
    if (org.mozdev.smoothwheel.ext.sw_isWindow) return item.innerHeight;
    return item.clientHeight;
},
sw_doScrollVertBy: function(target, y){
    var sw=org.mozdev.smoothwheel.ext;
    sw.sw_currentYPos+=y;
    if (sw.sw_isWindow){
        //target.scrollBy(0,y);
        //target.scrollTo(0, target.scrollY+y);
        //The above are the 'naive' implemenons, but in full-page-zoom, y is devided by the zoom factor, and if less than 1, scroll command is ignored.
        //on those cases, win.scrollY isn't updated as well. IMO it's a gecko bug.
        //This is a workaround to always scroll to an absolute position by registering the initial position and updating it as we scroll
        target.scrollTo(target.scrollX, sw.sw_currentYPos);//working
    }
    else target.scrollTop = (sw.sw_currentYPos);
},



sw_savedStatusNode:null,
sw_savedStatusTimeoutId:0,
sw_restoreStatus: function(){
    try{org.mozdev.smoothwheel.ext.sw_savedStatusNode.status="";}
    catch(e){
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_restoreStatus", e);
    };
    sw_savedStatusNode=null;
},


sw_doRequiredVisit: function(){with(org.mozdev.smoothwheel.ext){
    try{
        sw_debugMode&&dbg("!needs visit: "+sw_needsVisit);
        var nv=sw_needsVisit;
        sw_needsVisit="";
        sw_prefServices.setCharPref(sw_PrefExtensionRoot+"needs_visit","");
        var ua=(""+window.navigator.userAgent).toLowerCase();
        sw_debugMode&&dbg("!ua=: "+ua);
        if (ua.indexOf("firefox")>0 || ua.indexOf("seamonkey")>0 || ua.indexOf("songbird")>0)
            //not tested extensively on other systems
            sw_openBrowserWindow(nv);
        else
            sw_debugMode&&dbg("!Not firefox/seamonkey/songbird, not opening welcome screen");

        return;
    } catch(e){
        sw_needsVisit="";
        sw_prefServices.setCharPref(sw_PrefExtensionRoot+"needs_visit","");
        org.mozdev.smoothwheel.ext.sw_exceptionDbg("sw_doRequiredVisit", e);
    }
}},

sw_isAlreadyScrolling:false,
//SmoothWheel wheel event handler, registered for handling DOMMouseScroll and keydown events
smoothWheelHandler: function(e){with (org.mozdev.smoothwheel.ext){
//    sw_debugMode&&dbg("!e");
    if (!sw_isInitialized)
        return;
    if (e.sw_handled)
        return;

    e.sw_handled=true;
    
    if (sw_isAlreadyScrolling){
        //already scrolling. check 1st that we're on the same window
        if (!sw_isInParentsTree(e.target, sw_clientFrame, sw_isWindow)){
            sw_debugMode&&dbg("!Not same window. stopping scroll, aborting stats");
            sw_stopScroll();
        }
    }

    if (!sw_isAlreadyScrolling){
        var prefsRead=false;
        if (e.type=="keydown" && e.shiftKey && e.ctrlKey){//check enable/disable toggle
            sw_readPrefs(); prefsRead=true;
            if (sw_kbToggle_DOM_VK!=0 && e.keyCode==sw_kbToggle_DOM_VK){//check enable/disable toggle
                sw_setPref("bool", sw_PrefExtensionRoot+"quickDisable", sw_EnableThisExtension);
                var str="SmoothWheel is now "+((!sw_EnableThisExtension)?"On":"Off");
                sw_debugMode&&dbg("!"+str);
                try{
                    var win=e.target.ownerDocument.defaultView;
                    sw_savedStatusNode=win;
                    win.status=str;
                    clearTimeout(sw_savedStatusTimeoutId);
                    sw_savedStatusTimeoutId=setTimeout ("org.mozdev.smoothwheel.ext.sw_restoreStatus()", 2500);
                }catch(e){
                    org.mozdev.smoothwheel.ext.sw_exceptionDbg("smoothWheelHandler - set win status text", e);
                };
                return;
            }
        }
        
        if (e.type=="keydown"&& e.keyCode!=e.DOM_VK_DOWN && e.keyCode!=e.DOM_VK_UP)//abort key events which aren't down or up arrows
            return false;
        
        if (!prefsRead) sw_readPrefs();
        
        if  (!sw_EnableThisExtension){
            return;
        }

        if (sw_needsVisit!=""){
            return sw_doRequiredVisit();
        }
    }

    //potential scroll event
    if (!sw_acquireEvent(e, sw_isAlreadyScrolling)){ //if event should not be intercepted
        //sw_debugMode&&dbg("-");
        return;
    }

    e.preventDefault();
    e.stopPropagation();
    
    if (!sw_isAlreadyScrolling && sw_scrollLeft!=0){//initialize scroll
        sw_debugMode&&dbg("!initializing scroll");
        sw_initScrollLoop();
    }
}},

sw_statsTimeoutId:0,
sw_initScrollLoop: function(){with (org.mozdev.smoothwheel.ext){
    clearTimeout(sw_statsTimeoutId);
    sw_stopScroll(true);
    sw_isAlreadyScrolling=true;
    
    if (sw_ScrollMaxFPS==0){//brute force (0ms wait one shots)
        sw_debugMode&&dbg("!starting as-fast-as-possible");
        return sw_nextLoopIteration();
    }
        
    //implicit else: fixed frequency
  
    if (sw_timerMode==0){//interval pure javascript
        sw_debugMode&&dbg("!starting legacy timer ("+sw_LoopInterval+"ms)");
        return (sw_intervalID= setInterval(sw_smoothWheelLoop,sw_LoopInterval));
    }
        
    //implicit else: //nsITimer
    sw_debugMode&&dbg("!starting timer ("+sw_LoopInterval+"ms)");
    sw_nsITimerRepeat(sw_smoothWheelLoop, sw_LoopInterval);
}},
sw_stopScroll:function(isAvoidLog){
    var sw=org.mozdev.smoothwheel.ext;
//    sw.sw_debugMode&&sw.dbg("!stopping scroll")
    window.clearInterval(sw.sw_intervalID);
    window.clearTimeout(sw.sw_timeoutID);
    try{
        if (sw.sw_timerMode==1)
            sw.sw_nsITimer.cancel();
    }catch(e){
        sw.sw_debugMode&&sw.dbg("!!ERROR: couldn't cancel nsITimer");
    }
    sw.sw_isAlreadyScrolling=false;
    sw.sw_velocityCurrent=0;//sw_velocityLastEvent uses this as a start value. if incorrect, it'll affect scroll
    (!isAvoidLog)&&sw.sw_debugMode&&sw.dbg("!scroll stopped")
},

sw_nsITimer: null,
sw_nsITimerOneShot: function (func, delay){
    if (arguments.length==1) delay=1;
    org.mozdev.smoothwheel.ext.sw_nsITimer.initWithCallback({notify:func}, delay, Components.interfaces.nsITimer.TYPE_ONE_SHOT);
},
sw_nsITimerRepeat: function (func, interval){
    if (arguments.length==1) interval=1;
    //TYPE_REPEATING_PRECISE mode is too dangerous on slow pages (all events are sent). slack mode stops events untill the handler is done.
    org.mozdev.smoothwheel.ext.sw_nsITimer.initWithCallback({notify:func}, interval, Components.interfaces.nsITimer.TYPE_REPEATING_SLACK);
},
sw_nextLoopIteration: function(){with (org.mozdev.smoothwheel.ext){
    if (sw_ScrollMaxFPS>0){//repeat timer mode. do nothing
        //sw_debugMode&&dbg("freq-none");
        return;
    }
    //implicit else: trigger a one-shot
    
    if (sw_timerMode==0){//javascript
        //sw_debugMode&&dbg("one-shot javascript timeout ("+0+")");
        return setTimeout("org.mozdev.smoothwheel.ext.sw_smoothWheelLoop()",0);
    }

    //implicit else: nsITimer
    //sw_debugMode&&dbg("one-shot nsITimer ("+0+")");
    sw_nsITimerOneShot(sw_smoothWheelLoop,0);
}},


sw_lastWheelLoopEvent: 0,
sw_fpsStart:0, sw_fpsCounter1:0, sw_fpsCounter2:0, sw_fpsCounter3:0,
//loop for smooth scrolling the page
sw_smoothWheelLoop: function(){with (org.mozdev.smoothwheel.ext){
    if (!sw_isAlreadyScrolling){
        sw_debugMode&&dbg("!!ERROR: arrived at handler when shouldn't. abort scroll");
        return sw_stopScroll()
    }
    var toScroll, currentTime=(new Date()).getTime(), currentPercentage, timeLeft;

    if(sw_debugMode){
        var fd=currentTime-sw_fpsStart;
        if (sw_velocityCurrent==0 || fd>500 && fd<2000){
            if (fd>500){
                sw_debugMode&&dbg("!Rates ("+fd+"ms): A="+Math.round(1000*sw_fpsCounter1/fd)+" B="+Math.round(1000*sw_fpsCounter2/fd)+" C="+Math.round(1000*sw_fpsCounter3/fd));
            }
            
            sw_fpsStart=currentTime;
            sw_fpsCounter1=sw_fpsCounter2=sw_fpsCounter3=1;
            
        }
    }
    
    sw_fpsCounter1++;

    if (currentTime==sw_lastWheelLoopEvent)
        return sw_nextLoopIteration();

    sw_fpsCounter2++;
    sw_lastWheelLoopEvent= currentTime;
    
    if (!sw_clientFrame||isNaN(sw_scrollDeadline)||isNaN(sw_scrollCurrentDuration)||isNaN(sw_scrollLeftLastEvent)||isNaN(sw_velocityCurrent)||isNaN(sw_scrollLeft))
        return sw_panicAbort();

    timeLeft= sw_scrollDeadline - currentTime;

    if (timeLeft<=0)  //we're past the deadline. scroll what's left.
        toScroll=sw_scrollLeft;
    else{ //calculate next scroll destination according to scroll profile
        currentPercentage= 1 - timeLeft/sw_scrollCurrentDuration;
        toScroll= Math.round (sw_scrollLeftLastEvent*(sw_scrollProfile(currentPercentage) - 1) + sw_scrollLeft);
        if ((sw_scrollLeft-toScroll)*sw_scrollLeft < 0) //if going past destination point
            toScroll=sw_scrollLeft;
    }

    if ((toScroll==0) && (sw_scrollLeft!=0))
        return sw_nextLoopIteration();

    sw_fpsCounter3++;
    try{
        sw_doScrollVertBy(sw_clientFrame, toScroll);
        sw_debugMode&&dbg(""+toScroll+"px", true);
    } catch(e){
        sw_debugMode&&dbg("!!ERROR: can't scroll. abort.");
        return sw_stopScroll();
    }
    sw_scrollLeft -= toScroll;

    if (sw_scrollLeft==0){//no more scroll needed
        sw_stopScroll();
        sw_statsTimeoutId=setTimeout("org.mozdev.smoothwheel.ext.sw_saveStats()", 250);
        var fd=currentTime-sw_fpsStart;
        sw_debugMode&&dbg("!Rates: A="+Math.round(1000*sw_fpsCounter1/fd)+" B="+Math.round(1000*sw_fpsCounter2/fd)+" C="+Math.round(1000*sw_fpsCounter3/fd));
        return;
    }
    
    sw_nextLoopIteration();
}},

//the secret sauce that transform percentage of time to percentage of location
//it uses and updates an implicit parameter of the speed of the scroll at the time the event was intercepted,
//and updates current velocity
sw_scrollProfile: function(p){
    var sw_p1, sw_p2, sw_p21, sw_p2v, sw=org.mozdev.smoothwheel.ext;
    sw_p1=1-p; sw_p2=sw_p1*sw_p1; sw_p21=1-sw_p2; sw_p2v=sw_p2*sw.sw_velocityLastEvent;
    sw.sw_velocityCurrent= sw_p2v + sw_p21*2*sw_p1;
    return sw_p2v*p + sw_p21*sw_p21;
},

sw_isInitialized:false,
//initialize variable according to the preset, and determine whether optimization is possible.
sw_initSmoothWheel: function(){with (org.mozdev.smoothwheel.ext){
    try{
        sw_initGlobals();
        sw_readPrefs();
        sw_timerMode=0;
        sw_canUse_nsITimer_result=sw_canUse_nsITimer();

        if (sw_canUse_nsITimer())
            sw_timerMode=1;
        sw_isInitialized=true;
        sw_debugMode&&dbg("!SmoothWheel: initialized");
    } catch(e){
        sw_isInitialized=false;
        sw_debugMode&&dbg("!! ERROR: unable to initialize SmoothWheel");
    }
}},


sw_canUse_nsITimer: function(){
    var sw=org.mozdev.smoothwheel.ext;
    try{
        sw.sw_nsITimer=Components.classes["@mozilla.org/timer;1"].createInstance(Components.interfaces.nsITimer);
        sw.sw_nsITimer.cancel();
        if (!sw.sw_nsITimer || 
            !sw.sw_nsITimer.cancel || 
            !sw.sw_nsITimer.initWithCallback || 
            undefined==Components.interfaces.nsITimer.TYPE_ONE_SHOT ||
            //undefined==Components.interfaces.nsITimer.AVIH_NONEXIST_CONST ||
            undefined==Components.interfaces.nsITimer.TYPE_REPEATING_SLACK ||
            sw.sw_nsITimer.cancel()
            ){
            
            sw.sw_debugMode&&sw.dbg("!! nsITimer ompatibility(1): Failed");
            sw.sw_nsITimer=null;
            return false;
        }
    } catch(e){
        sw.sw_debugMode&&sw.dbg("!! nsITimer ompatibility(2): Failed ("+e+")");
        sw.sw_nsITimer=null;
        return false;
    }
    
    sw.sw_debugMode&&sw.dbg("! OK, can use nsITimer");    
    return true;
},

// ********** start: debug/test functions *************
sw_outMsg: function(t){
    try{Components.utils.reportError(t);}catch(e){};
},
sw_outError: function(t){
    try{
    var consoleService = Components.classes["@mozilla.org/consoleservice;1"].getService(Components.interfaces.nsIConsoleService);
    consoleService.logStringMessage(t);//aparently doesn't work
    }catch(e){}
},

sw_outLog: function(t){
    var sw=org.mozdev.smoothwheel.ext;
    sw.sw_logCurrent+=t+"\r\n";
},

sw_ecounter: 0,
sw_lastDbg:"",
sw_lastDbgTimestamp:0,
sw_noNewLineDbg:"",
// display a debug text on the javascript console.
//sw_debugMode can be set to 0/1/2/3 for debug levels (higher=more). on older gecko (3.5.x-?). add 8 (for 9/10/11) to report as error instead of message
dbg: function(t, isAvoidNL){with (org.mozdev.smoothwheel.ext){
//    return;
    var outf=sw_outMsg;
    if (sw_debugMode&&8)
        outf=sw_outError;
    if (sw_logDebug)
        outf=sw_outLog;
        
    t=arguments[0];
    var dm=sw_debugMode&7;
    if (dm==0 || dm==2&&t.indexOf("!")!=0 || dm==1&&t.indexOf("!!")!=0 ){
        sw_lastDbg=t;
        return;
    } else{
        if (sw_lastDbg!="")
           outf("[auto-prev]"+sw_lastDbg); 
    }
    sw_lastDbg="";
    var str=t;
    try{
        var now=(new Date()).getTime();
        
        if (isAvoidNL){
            if (sw_noNewLineDbg!="") sw_noNewLineDbg+="/";
            sw_noNewLineDbg+=""+(now-sw_lastDbgTimestamp)+"ms:"+ str;
            sw_ecounter++;
        } else {
            if (sw_noNewLineDbg!=""){
                outf("#"+(sw_ecounter-1)+" "+sw_noNewLineDbg);
                sw_noNewLineDbg="";
            }
            var out="#"+(sw_ecounter++)+" ("+(now-sw_lastDbgTimestamp)+"ms): "+ str;
            outf(out);
        }
        sw_lastDbgTimestamp=now;
    } catch(e){};
}}

// ********** end: debug/test functions *************


};//org.mozdev.smoothwheel.ext

//runs once, when the file is loaded.
org.mozdev.smoothwheel.ext.sw_initSmoothWheel();


/***************************************************************************
enhanced changelog for early versions:

/////////////////////////////////////////////////////////////////////////////
v0.44.5.20050523 (2005-05-23)
- added more variables to the configuration panel
- modified prefutils a bit to allow foating point values (as strings)
- more options exposed, better usability by rearrangement of settings and about panel.
- clean preferences when upgrading a version
/////////////////////////////////////////////////////////////////////////////

v0.43.1.20041107 (2004-11-07)
- max-version reduced such that it's accepted by update.mozilla.org
- added build date to the version number
- added tooltip for the modifiers section in the configuration panel

/////////////////////////////////////////////////////////////////////////////
v0.43 (2004-11-07)

improvements (v0.43):
- Added configuration dialog for Firefox/Thunderbird  (from the extensions panel)
  - in mozilla/netscape, enter address: chrome://smoothwheel/content/prefoverlay.xul

/////////////////////////////////////////////////////////////////////////////
v0.42 (2004-10-22)

improvements (v0.42):
- Added support for Firefox-1.0 and Thunderbird-1.0
  - max supported versions: FF: 5.0, TB: 5.0 (I've had it with max-version)

/////////////////////////////////////////////////////////////////////////////
v0.41 (2004-07-03)

improvements (v0.41):
- Modified to also support Firefox-0.9 and Thunderbird-0.7 new Extension-Manager API
  - max supported versions: FF: 0.999, TB: 0.799 (due to possible more EM API changes)

/////////////////////////////////////////////////////////////////////////////
v0.4 (2004-02-04)

bugFix (v0.4):
- Refined algorithms implementation (a bit more responsive)
- Fixed: occasional misbehaviour if not using a preset.

improvements (v0.4):
- Averaged mouse wheel events (better predictability and stability)

/////////////////////////////////////////////////////////////////////////////
v0.33.1 (2004-02-01)

bugFix (v0.33.1):
- Removed 'handled(normal)' alert if default settings are changed.

/////////////////////////////////////////////////////////////////////////////
v0.33 (2004-01-27)

bugFix (v0.33):
- Improved compatibility with latest Moz abd Birds releases (detection code from AIO v0.96)

/////////////////////////////////////////////////////////////////////////////
v0.32 (2003-08-02)

bugFix (v0.32):
- Works in Mozilla-Thunderbird and Help window (Mozilla/Netscape).
- Improved compatibility with Mozilla-Firebird 0.6.1 and Mozilla-Thunderbird 0.1.

/////////////////////////////////////////////////////////////////////////////
v0.31 (2003-06-05)

bugFix (v0.31):
- Fixed a compatibility issue with Scroll-Wheel-Navigation at All-In-One-Gestures extension.

/////////////////////////////////////////////////////////////////////////////
v0.3 (22-May-2003)

bugFix (v0.3):
- new target detection code, courtesy of All-In-One-Gestures author.
- cleanup of namespace (precaution measures)

new features (v0.3):
- now working on view source (all browsers) and composer.
- modifier key for temporarily disabling SmoothWheel.
- Soft-Edge : Lands gracefully on top/bottom of document.
- Adaptive-Step : Shorter duration between wheel events -> more pixels/event.
- Adaptive-Duration : Shorter duration between wheel events -> scrolls faster.
- more optimizations for event handler if default settings are kept.
- some more presets to reflect the new features.

status (v0.3):

scrolling SMOOTHLY on:
- mozilla: browser, forms, view-source, mail-and-news (message windows), composer, irc (main window)
- phoenix/firebird: browser, forms, view-source

letting default scroll handler do it's stuff on:
- mozilla: top/left panes of mail-and-news, left/bottom irc panes, all XUL panes, about:config
- phoenix/firebird: about:config, all XUL panes

breaking wheel scroll functionality on (v0.2):
- couldn't find any so far.

/////////////////////////////////////////////////////////////////////////////
v0.2 (15-Apr-2003)

bugFix (v0.2):
- much better target detection code. will let default scroll handler do it's stuff if we can't handle properly.
- better timing of scroll loop
- fps limit (for fast CPUs)

new features (v0.2):
- modifier keys for slow/fullPage scroll
- improved algorithm: smooth transition if scroll event occures while still scrolling
- much more configuration options (including 6 presets)
- optimized code for the most common settings


status (v0.2):

scrolling SMOOTHLY on:
- mozilla: browser, view-source, irc (main window), mail-and-news (message windows)
- phoenix: browser

letting default scroll handler do it's stuff on:
- mozilla: browser forms, some mail-and-news panes, some irc panes,
           composer, all XUL panes, about:config, more..
- phoenix: browser forms, view-source, about:config, all XUL panes, more..

breaking wheel scroll functionality on (v0.2):
- couldn't find any so far.

/////////////////////////////////////////////////////////////////////////////
v0.1 (22-Mar-2003)

status (v0.1):

scrolling SMOOTHLY on:
- mozilla: browser, view-source, irc (main window), mail-and-news (message windows)
- phoenix: browser

letting default scroll handler do it's stuff on:
- mozilla: composer
- phoenix: view-source

breaking wheel scroll functionality on:
- mozilla: browser forms, some mail-and-news panes, some irc panes, all XUL panes, more...
- phoenix: browser forms, all XUL panes, more...
***************************************************************************
*/
